#ifndef STEMMGR_HPP
#define STEMMGR_HPP

#include "grammar.hpp"

#include <map>

#ifdef __BORLANDC__
typedef std::map<std_string, Grammar*, std::less<std_string> > StringToGrammarDict;
#else
typedef map<std_string, Grammar*, less<string> > StringToGrammarDict;
#endif


const int kNumberOfStemTypes=39;
const int kNumberOfOrtographies=2;
const int kNoOrt=-1;

class StemManager;

class StemNode
{
   friend class StemManager;
public:
   StemNode(const std_string &stemCode);
   ~StemNode();
   std_string value() const;
   std_string pureValue() const;
   int hasExceptionalTransition() const { return fExceptionalTransition; }
   int ortography() const { return fOrtNr; }
   int isIntermediate() const { return fIntermediate; }
private:
   StemNode* createSibling(const std_string &stemCode, int ortNr=-1,
                           int bExceptionalTransition=0, int bIntermediate=0);
   StemNode* createChild(const std_string &stemCode, int ortNr=-1,
                         int bExceptionalTransition=0, int bIntermediate=0);
   StemNode* findStemNode(const std_string &stemCode);
   std_string fValue;
   StemNode *fpFirstChild;
   StemNode *fpSibling;
   StemNode *fpParent;
   int fOrtNr;
   char fExceptionalTransition:1;
   char fIntermediate:1;
};

class StemManager
{
public:
   StemManager();
   ~StemManager();
   StemManager& loadFromFile(const char * const fileName);
   const StemNode* previousStemNode(const StemNode *pNode);
   const StemNode* getStemNode(int index, const std_string &stemCode);
   StemManager& addGrammar(const std_string &grammarName, const std_string &grammarFile,
                           RulePrelim *pStaticRules, int nrStaticRules,
                              ExceptionRegistry *pExcRegistry, int baseIsLeft);
   void transform(ResultCollection *pResults, DerivationResult *pInput,
                  const std_string &grammarName, const std_string &targetStemCode,
                  int stemNr, int bExceptionalTransition, int withAppostr, int *msg);
   void disAmbiguate(ResultCollection *pResults, DerivationResult *pInput,
                     const std_string &grammarName, const std_string &targetStemCode,
                     int stemNr, int bExceptionalTransition, int withAppostr, int *msg);
   StemManager& unloadGrammars();
   ResultCollection* createLemma(const std_string &inString, int stemNr,
                                 const std_string &stemCode, int withAppostr=0);
   ResultCollection* createForms(const std_string &inString, int stemNr, int withAppostr=0);
   StemManager& setOrtography(int ortNr, const std_string &grammarName);
   const Grammar* findGrammar(const std_string &grammarName);
private:
   StemNode *firstStem[kNumberOfStemTypes];
   StringToGrammarDict fGrammarDict;
   std_string fOrtography[kNumberOfOrtographies];
};

#endif
