#ifdef LINUX_PORT
#define _stdcall
#else
#include <windows.h>
#endif

#include "tdexc.hpp"

//#include <list>
#include<deque>
#include<strndef.hpp>
#include <string.h>
#include <stdio.h>

#include "patmatch_exp.h"
#include <syll_exp.h>
#include <openfile.h>
#include <appostr.hpp>


class TypeDetRule;

#ifdef __BORLANDC__
typedef std::deque<TypeDetRule*> TypeDetRulesList;
#else
typedef deque<TypeDetRule*> TypeDetRulesList;
#endif



class TypeDetRule
{
public:
  TypeDetRule();
  ~TypeDetRule();
  TypeDetRule(const char *pat, const char *res);
//  TypeDetRule(const TypeDetRule &other);
//  TypeDetRule& operator=(const TypeDetRule &other);
  void addSubRule(TypeDetRule *pSub);
  const char* test(const char *aWord) const;

private:
  int numberOfSigns() const;
  char fPattern[16];
  std_string fResult;
  int fCompLength;
  TypeDetRulesList fSubRules;
};

/***************************************************************************/

struct TypeDetRulePrelim
{
  char fNumSyll;
  char fPattern[12];
  char fResult[12];
  char fSub;
};

TypeDetRulePrelim gNounRulesPrelim[]= {
  { '\0', "W", "", '\0'  },
  { '\0', "W1W1", "26_S", '\1' },
  { '\0', "C", "", '\0'  },
  { '\0', "Wh", "41_I", '\1' },
  { '\0', "j", "#", '\1' },
  { '\0', "z", "#", '\1' },
  { '\1', "W", "", '\0'  },
  { '\1', "WA", "26_S", '\1' },
  { '\1', "!WW", "#", '\1' },
  { '\1', "CC", "", '\0'  },
  { '\1', "W1W2RG", "02_S", '\1' },
  { '\1', "CC", "22_S", '\1' },
  { '\1', "WC", "", '\0'  },
  { '\1', "Wis", "22_S", '\1' },
  { '\1', "W1W2s", "09_S", '\1' },
  { '\1', "WWC", "22_S", '\1' },
  { '\1', "!WWC", "#", '\1' },
  { '\2', "WW", "", '\0'  },
  { '\2', "WUe", "06_S", '\1' },
  { '\2', "WUE", "16_S", '\1' },
  { '\2', "CWA", "26_S", '\1' },
  { '\2', "ne", "", '\0'  },
  { '\2', "WWne", "10_A", '\1' },
  { '\2', "!WWRne", "06_S", '\1' },
  { '\2', "Cne", "02_A", '\1' },
  { '\2', "Ce", "", '\0'  },
  { '\2', "WWke", "12_S?10_S", '\1' },
  { '\2', "WWCe", "06_S", '\1' },
  { '\2', "!WWZe", "02_A", '\1' },
  { '\2', "!WWGle", "16_S", '\1' },
  { '\2', "Wsse", "#", '\1' },
  { '\2', "Ce", "06_S", '\1' },
  { '\2', "Ci", "", '\0'  },
  { '\2', "Ksti", "41_D", '\1' },
  { '\2', "hti", "41_D", '\1' },
  { '\2', "!WWZi", "17_S", '\1' },
  { '\2', "Ci", "16_S", '\1' },
  { '\2', "Ca", "", '\0'  },
  { '\2', "!WWZa", "17_S", '\1' },
  { '\2', "WWja", "01_S", '\1' },
  { '\2', "WKja", "01_S", '\1' },
  { '\2', "WWCCa", "01_S", '\1' },
  { '\2', "CCCa", "01_S", '\1' },
  { '\2', "Ca", "16_S", '\1' },
  { '\2', "Cu", "", '\0'  },
  { '\2', "!WWZu", "17_S", '\1' },
  { '\2', "WWtu", "01_A", '\1' },
  { '\2', "WWCCu", "01_S", '\1' },
  { '\2', "Cu", "16_S", '\1' },
  { '\2', "Co", "", '\0'  },
  { '\2', "!WWZo", "17_S~16_S", '\1' },
  { '\2', "Co", "16_S", '\1' },
  { '\2', "CO", "#", '\0' },
  { '\2', "WWC", "", '\0'  },
  { '\2', "Wius", "11_S~09_S", '\1' },
  { '\2', "WW1W2C", "02_S", '\1' },
  { '\2', "Eim", "02_A", '\1' },
  { '\2', "WWC", "22_S", '\1' },
  { '\2', "Cus", "", '\0'  },
  { '\2', "!WWGus", "02_A", '\1' },
  { '\2', "!WWHus", "09_S", '\1' },
  { '\2', "!WWKus", "09_S", '\1' },
  { '\2', "Cus", "11_S~09_S", '\1' },
  { '\2', "Cas", "", '\0'  },
  { '\2', "Cjas", "02_A", '\1' },
  { '\2', "WWjas", "02_A", '\1' },
  { '\2', "WWkas", "02_A", '\1' },
  { '\2', "Cas", "07_S", '\1' },
  { '\2', "CAs", "", '\0'  },
  { '\2', "Ces", "09_S", '\1' },
  { '\2', "rssis", "41_DA", '\1' },
  { '\2', "KKis", "41_DA", '\1' },
  { '\2', "WLKis", "41_DA", '\1' },
  { '\2', "WWKis", "11_S~09_S", '\1' },
  { '\2', "Cis", "09_S", '\1' },
  { '\2', "Cos", "11_S~09_S", '\1' },
  { '\2', "Cud", "", '\0'  },
  { '\2', "Wnud", "06_A", '\1' },
  { '\2', "Dud", "02_A", '\1' },
  { '\2', "CAQ", "", '\0'  },
  { '\2', "Clik", "25_A", '\1' },
  { '\2', "CCnik", "25_S", '\1' },
  { '\2', "WWCnik", "25_S", '\1' },
  { '\2', "Cet", "#", '\1' },
  { '\2', "CAQ", "02_S", '\1' },
  { '\2', "CWC", "", '\0'  },
  { '\2', "CAv", "02_A", '\1' },
  { '\2', "CAm", "02_A", '\1' },
  { '\2', "kvel", "41_DA", '\1' },
  { '\2', "Cil", "#", '\1' },
  { '\2', "CWM", "02_S", '\1' },
  { '\2', "CAG", "02_S", '\1' },
  { '\2', "CWC", "#", '\1' },
  { '\2', "CC", "", '\0'  },
  { '\2', "oCG", "22_S", '\1' },
  { '\2', "CG", "02_S", '\1' },
  { '\2', "sk", "02_S", '\1' },
  { '\2', "Elt", "41_D", '\1' },
  { '\2', "Wlt", "#", '\1' },
  { '\2', "!WWCist", "22_S", '\1' },
  { '\2', "Ast", "#", '\1' },
  { '\2', "Aks", "#", '\1' },
  { '\2', "WWKs", "22_S", '\1' },
  { '\2', "NKs", "22_S", '\1' },
  { '\2', "Ks", "02_S", '\1' },
  { '\2', "Rs", "02_S", '\1' },
  { '\2', "CC", "22_S", '\1' },
  { '\3', "WW", "", '\0'  },
  { '\3', "ia", "01_S", '\1' },
  { '\3', "io", "01_S", '\1' },
  { '\3', "eo", "01_S", '\1' },
  { '\3', "WA", "26_S", '\1' },
  { '\3', "Ca", "", '\0'  },
  { '\3', "W1W1Ca", "16_S", '\1' },
  { '\3', "Wika", "16_S", '\1' },
  { '\3', "ika", "01_S", '\1' },
  { '\3', "Aja", "01_S", '\1' },
  { '\3', "ldasa", "41_D", '\1' },
  { '\3', "mata", "41_A", '\1' },
  { '\3', "Wta", "#", '\1' },
  { '\3', "Wna", "#", '\1' },
  { '\3', "Wga", "#", '\1' },
  { '\3', "Wda", "#", '\1' },
  { '\3', "W1W1CCa", "16_S", '\1' },
  { '\3', "Nja", "16_S", '\1' },
  { '\3', "nna", "16_S", '\1' },
  { '\3', "Mka", "16_S", '\1' },
  { '\3', "Fka", "16_S", '\1' },
  { '\3', "lla", "16_S", '\1' },
  { '\3', "lda", "#", '\1' },
  { '\3', "Ca", "01_S", '\1' },
  { '\3', "Cne", "02_A", '\0' },
  { '\3', "lane", "", '\0'  },
  { '\3', "!WWZlane", "10_A", '\1' },
  { '\3', "Tlane", "12_S?10_S", '\1' },
  { '\3', "WWlane", "12_S?10_S", '\1' },
  { '\3', "Wlane", "10_A", '\1' },
  { '\3', "line", "", '\0'  },
  { '\3', "WWline", "12_A?10_A", '\1' },
  { '\3', "Wline", "10_A", '\1' },
  { '\3', "!WWZline", "10_A", '\1' },
  { '\3', "Tline", "12_A~10_A", '\1' },
  { '\3', "mine", "", '\0'  },
  { '\3', "tmine", "12_S?10_S", '\1' },
  { '\3', "smine", "12_S?10_S", '\1' },
  { '\3', "dmine", "12_S?10_S", '\1' },
  { '\3', "Wmine", "10_A", '\1' },
  { '\3', "!WWZmine", "10_A", '\1' },
  { '\3', "Tmine", "12_A?10_A", '\1' },
  { '\3', "kene", "", '\0'  },
  { '\3', "WWkene", "12_S?10_S", '\1' },
  { '\3', "Ckene", "12_A~10_A", '\1' },
  { '\3', "kene", "10_A", '\1' },
  { '\3', "Ane", "", '\0'  },
  { '\3', "WWQAne", "12_A~10_A", '\1' },
  { '\3', "WCAne", "10_A", '\1' },
  { '\3', "WWKsAne", "12_A~10_A", '\1' },
  { '\3', "KsAne", "10_A", '\1' },
  { '\3', "WWCCAne", "12_A~10_A", '\1' },
  { '\3', "WZZAne", "10_A", '\1' },
  { '\3', "WhvKAne", "10_A", '\1' },
  { '\3', "WUEne", "10_A", '\1' },
  { '\3', "Ane", "12_A~10_A", '\1' },
  { '\3', "Ce", "", '\0'  },
  { '\3', "WW1W2ke", "12_S", '\1' },
  { '\3', "WWke", "12_S?10_S", '\1' },
  { '\3', "Ake", "12_S", '\1' },
  { '\3', "W1W1Ce", "16_S", '\1' },
  { '\3', "kvele", "41_D", '\1' },
  { '\3', "Ale", "#", '\1' },
  { '\3', "Wde", "#", '\1' },
  { '\3', "Wte", "#", '\1' },
  { '\3', "Wme", "#", '\1' },
  { '\3', "Wge", "#", '\1' },
  { '\3', "Nje", "16_S", '\1' },
  { '\3', "lge", "#", '\1' },
  { '\3', "Wsse", "#", '\1' },
  { '\3', "Akse", "#", '\1' },
  { '\3', "Ce", "01_S", '\1' },
  { '\3', "sti", "", '\0'  },
  { '\3', "asti", "41_D", '\1' },
  { '\3', "esti", "41_D", '\1' },
  { '\3', "Asti", "01_S", '\1' },
  { '\3', "Ksti", "41_D", '\1' },
  { '\3', "Ci", "", '\0'  },
  { '\3', "W1W1Ci", "16_S", '\1' },
  { '\3', "uti", "01_S", '\1' },
  { '\3', "Ati", "41_D", '\1' },
  { '\3', "CAli", "41_D", '\1' },
  { '\3', "Asi", "41_D", '\1' },
  { '\3', "Avi", "41_D", '\1' },
  { '\3', "Ani", "#", '\1' },
  { '\3', "Wgi", "#", '\1' },
  { '\3', "Ndi", "01_S", '\1' },
  { '\3', "Atsi", "41_D", '\1' },
  { '\3', "hti", "41_D", '\1' },
  { '\3', "Vgi", "#", '\1' },
  { '\3', "Cki", "#", '\1' },
  { '\3', "Ci", "01_S", '\1' },
  { '\3', "Co", "16_S", '\0' },
  { '\3', "tu", "", '\0'  },
  { '\3', "datu", "01_S", '\1' },
  { '\3', "tatu", "01_S", '\1' },
  { '\3', "Atu", "01_A", '\1' },
  { '\3', "stu", "01_S", '\1' },
  { '\3', "Cu", "", '\0'  },
  { '\3', "Anu", "01_S", '\1' },
  { '\3', "elu", "17_S", '\1' },
  { '\3', "Agu", "#", '\1' },
  { '\3', "aku", "#", '\1' },
  { '\3', "stikku", "41_D", '\1' },
  { '\3', "ldu", "01_S", '\1' },
  { '\3', "lnu", "01_S", '\1' },
  { '\3', "lgu", "#", '\1' },
  { '\3', "Cu", "01_S", '\1' },
  { '\3', "CW", "#", '\0' },
  { '\3', "WWC", "", '\0'  },
  { '\3', "Aier", "02_S", '\1' },
  { '\3', "Eim", "02_A", '\1' },
  { '\3', "CW1W2L", "19_S~02_S", '\1' },
  { '\3', "Ius", "11_S", '\1' },
  { '\3', "iEs", "11_S", '\1' },
  { '\3', "Eid", "#", '\1' },
  { '\3', "WWC", "22_S", '\1' },
  { '\3', "Cus", "", '\0'  },
  { '\3', "W1W1CCus", "11_S~09_S", '\1' },
  { '\3', "ikkus", "11_S~09_S", '\1' },
  { '\3', "sKlus", "11_S", '\1' },
  { '\3', "CCCus", "11_S~09_S", '\1' },
  { '\3', "W1W1Cus", "11_S~09_S", '\1' },
  { '\3', "Cus", "11_S", '\1' },
  { '\3', "Cas", "", '\0'  },
  { '\3', "jas", "02_A", '\1' },
  { '\3', "ikas", "02_S", '\1' },
  { '\3', "kas", "02_A", '\1' },
  { '\3', "Ngas", "02_S", '\1' },
  { '\3', "CAs", "", '\0'  },
  { '\3', "WWHis", "09_S", '\1' },
  { '\3', "Cis", "11_S", '\1' },
  { '\3', "CAs", "#", '\1' },
  { '\3', "CWM", "", '\0'  },
  { '\3', "!WWZinal", "41_D", '\1' },
  { '\3', "akil", "41_DA", '\1' },
  { '\3', "Avil", "41_DA", '\1' },
  { '\3', "Cil", "#", '\1' },
  { '\3', "del", "#", '\1' },
  { '\3', "tel", "#", '\1' },
  { '\3', "WWCel", "02_S", '\1' },
  { '\3', "CCel", "02_S", '\1' },
  { '\3', "Cel", "#", '\1' },
  { '\3', "CAl", "19_S", '\1' },
  { '\3', "CAm", "02_A", '\1' },
  { '\3', "CAv", "02_A", '\1' },
  { '\3', "Cer", "02_S", '\1' },
  { '\3', "Cor", "02_S", '\1' },
  { '\3', "CWN", "19_S", '\1' },
  { '\3', "CAQ", "", '\0'  },
  { '\3', "W1W1lik", "25_A", '\1' },
  { '\3', "W1W1Mik", "02_S", '\1' },
  { '\3', "lik", "25_A", '\1' },
  { '\3', "Mik", "25_S", '\1' },
  { '\3', "stik", "25_S", '\1' },
  { '\3', "W1W1dik", "02_S", '\1' },
  { '\3', "ndik", "25_N", '\1' },
  { '\3', "rdik", "25_N", '\1' },
  { '\3', "dik", "25_S", '\1' },
  { '\3', "Cik", "02_S", '\1' },
  { '\3', "CAt", "#", '\1' },
  { '\3', "CAG", "", '\0'  },
  { '\3', "Dud", "02_A", '\1' },
  { '\3', "CAd", "#", '\1' },
  { '\3', "CAb", "#", '\1' },
  { '\3', "CWC", "#", '\0' },
  { '\3', "WCC", "", '\0'  },
  { '\3', "Elt", "41_D", '\1' },
  { '\3', "Wlt", "#", '\1' },
  { '\3', "W1W1Cist", "#", '\1' },
  { '\3', "ist", "22_S", '\1' },
  { '\3', "KNAst", "22_S", '\1' },
  { '\3', "GNAst", "22_S", '\1' },
  { '\3', "Ast", "#", '\1' },
  { '\3', "ng", "02_S", '\1' },
  { '\3', "taks", "22_S", '\1' },
  { '\3', "Aks", "#", '\1' },
  { '\3', "W1W1ZWts", "02_S", '\1' },
  { '\3', "WCC", "22_S", '\1' },
  { '\3', "CCC", "", '\0'  },
  { '\3', "Wnts", "22_S", '\1' },
  { '\3', "Wnss", "22_S", '\1' },
  { '\3', "CCC", "#", '\1' },
  { '\4', "WW", "", '\0'  },
  { '\4', "ia", "01_S", '\1' },
  { '\4', "io", "01_S", '\1' },
  { '\4', "WA", "26_S", '\1' },
  { '\4', "Ca", "", '\0'  },
  { '\4', "W1W1Ca", "16_S", '\1' },
  { '\4', "Wika", "16_S", '\1' },
  { '\4', "ika", "01_S", '\1' },
  { '\4', "Aja", "01_S", '\1' },
  { '\4', "ldasa", "41_D", '\1' },
  { '\4', "mata", "41_A", '\1' },
  { '\4', "Wta", "#", '\1' },
  { '\4', "Wna", "#", '\1' },
  { '\4', "Wga", "#", '\1' },
  { '\4', "Wda", "#", '\1' },
  { '\4', "nna", "16_S", '\1' },
  { '\4', "Nja", "16_S", '\1' },
  { '\4', "lla", "16_S", '\1' },
  { '\4', "la", "01_S", '\1' },
  { '\4', "Ca", "16_S", '\1' },
  { '\4', "Cne", "02_A", '\0' },
  { '\4', "lane", "", '\0'  },
  { '\4', "WWClane", "12_S?10_S", '\1' },
  { '\4', "ondlane", "12_S?10_S", '\1' },
  { '\4', "WWlane", "12_S?10_S", '\1' },
  { '\4', "lane", "12_S", '\1' },
  { '\4', "line", "", '\0'  },
  { '\4', "WWline", "12_A?10_A", '\1' },
  { '\4', "line", "12_A", '\1' },
  { '\4', "mine", "", '\0'  },
  { '\4', "Amine", "12_S", '\1' },
  { '\4', "Cmine", "12_A", '\1' },
  { '\4', "kene", "", '\0'  },
  { '\4', "WW1W2kene", "12_S", '\1' },
  { '\4', "WWkene", "12_S?10_S", '\1' },
  { '\4', "kene", "12_S", '\1' },
  { '\4', "Ane", "", '\0'  },
  { '\4', "tikune", "10_A?12_A", '\1' },
  { '\4', "W1W1ZAne", "10_A", '\1' },
  { '\4', "WW1W2CCAne", "12_A", '\1' },
  { '\4', "WWCCAne", "12_A~10_A", '\1' },
  { '\4', "WWne", "16_S", '\1' },
  { '\4', "matane", "10_A", '\1' },
  { '\4', "Ane", "12_A", '\1' },
  { '\4', "Ce", "", '\0'  },
  { '\4', "WWke", "12_S?10_S", '\1' },
  { '\4', "Ake", "12_S", '\1' },
  { '\4', "W1W1Ce", "16_S", '\1' },
  { '\4', "akile", "41_D", '\1' },
  { '\4', "evile", "41_D", '\1' },
  { '\4', "Ale", "#", '\1' },
  { '\4', "Wde", "#", '\1' },
  { '\4', "Wte", "#", '\1' },
  { '\4', "Wme", "#", '\1' },
  { '\4', "Wge", "#", '\1' },
  { '\4', "Wsse", "#", '\1' },
  { '\4', "Akse", "#", '\1' },
  { '\4', "sti", "", '\0'  },
  { '\4', "asti", "41_D", '\1' },
  { '\4', "esti", "41_D", '\1' },
  { '\4', "Asti", "41_D", '\1' },
  { '\4', "Ci", "", '\0'  },
  { '\4', "WWCi", "16_S", '\1' },
  { '\4', "misi", "41_D", '\1' },
  { '\4', "kesi", "41_D", '\1' },
  { '\4', "mini", "41_D", '\1' },
  { '\4', "kuti", "41_D", '\1' },
  { '\4', "iti", "41_D", '\1' },
  { '\4', "Avi", "41_D", '\1' },
  { '\4', "Ani", "#", '\1' },
  { '\4', "Wgi", "#", '\1' },
  { '\4', "hti", "41_D", '\1' },
  { '\4', "Vgi", "#", '\1' },
  { '\4', "Cki", "#", '\1' },
  { '\4', "Co", "16_S", '\0' },
  { '\4', "tu", "", '\0'  },
  { '\4', "datu", "01_S", '\1' },
  { '\4', "tatu", "01_S", '\1' },
  { '\4', "eeritu", "01_S", '\1' },
  { '\4', "Atu", "01_A", '\1' },
  { '\4', "Cu", "", '\0'  },
  { '\4', "stikku", "41_D", '\1' },
  { '\4', "elu", "17_S", '\1' },
  { '\4', "Anu", "01_S", '\1' },
  { '\4', "Agu", "#", '\1' },
  { '\4', "CW", "#", '\0' },
  { '\4', "WWC", "", '\0'  },
  { '\4', "Aier", "02_S", '\1' },
  { '\4', "Eim", "02_A", '\1' },
  { '\4', "CW1W2L", "19_S~02_S", '\1' },
  { '\4', "Eid", "#", '\1' },
  { '\4', "WWC", "22_S", '\1' },
  { '\4', "Cus", "", '\0'  },
  { '\4', "WWCCus", "11_S~09_S", '\1' },
  { '\4', "ikkus", "11_S~09_S", '\1' },
  { '\4', "Ndus", "11_S", '\1' },
  { '\4', "W1W1MUstus", "11_S", '\1' },
  { '\4', "W1W1MAvlus", "11_S", '\1' },
  { '\4', "W1W1MUtus", "11_S", '\1' },
  { '\4', "W1W1MUvus", "11_S", '\1' },
  { '\4', "Cus", "11_S?09_S", '\1' },
  { '\4', "CAs", "", '\0'  },
  { '\4', "kas", "02_A", '\1' },
  { '\4', "jas", "02_A", '\1' },
  { '\4', "Cis", "09_S", '\1' },
  { '\4', "CAs", "#", '\1' },
  { '\4', "CWM", "", '\0'  },
  { '\4', "Cer", "02_S", '\1' },
  { '\4', "Cor", "02_S", '\1' },
  { '\4', "CAv", "02_A", '\1' },
  { '\4', "soikum", "02_S", '\1' },
  { '\4', "W1W1Tum", "02_S", '\1' },
  { '\4', "WCCum", "02_S", '\1' },
  { '\4', "likum", "02_A", '\1' },
  { '\4', "iKum", "19_S", '\1' },
  { '\4', "CAm", "02_A", '\1' },
  { '\4', "del", "#", '\1' },
  { '\4', "tel", "#", '\1' },
  { '\4', "aabel", "02_A", '\1' },
  { '\4', "WWCel", "02_S", '\1' },
  { '\4', "CCel", "02_S", '\1' },
  { '\4', "Cel", "#", '\1' },
  { '\4', "Cil", "#", '\1' },
  { '\4', "CWN", "19_S", '\1' },
  { '\4', "CAQ", "", '\0'  },
  { '\4', "W1W1lik", "25_A", '\1' },
  { '\4', "W1W1Mik", "02_S", '\1' },
  { '\4', "lik", "25_A", '\1' },
  { '\4', "Mik", "25_S", '\1' },
  { '\4', "stik", "25_S", '\1' },
  { '\4', "ndik", "25_N", '\1' },
  { '\4', "Cik", "02_S", '\1' },
  { '\4', "CAt", "#", '\1' },
  { '\4', "CAG", "", '\0'  },
  { '\4', "Dud", "02_A", '\1' },
  { '\4', "CAd", "#", '\1' },
  { '\4', "CAb", "#", '\1' },
  { '\4', "CWC", "#", '\0' },
  { '\4', "WCC", "", '\0'  },
  { '\4', "Elt", "41_D", '\1' },
  { '\4', "Wlt", "#", '\1' },
  { '\4', "W1W1Cist", "#", '\1' },
  { '\4', "ist", "22_S", '\1' },
  { '\4', "KNAst", "22_S", '\1' },
  { '\4', "GNAst", "22_S", '\1' },
  { '\4', "Ast", "#", '\1' },
  { '\4', "ng", "02_S", '\1' },
  { '\4', "taks", "22_S", '\1' },
  { '\4', "Aks", "#", '\1' },
  { '\4', "W1W1ZWts", "02_S", '\1' },
  { '\4', "WCC", "22_S", '\1' },
  { '\4', "CCC", "", '\0'  },
  { '\4', "Wnts", "22_S", '\1' },
  { '\4', "Wnss", "22_S", '\1' },
  { '\4', "CCC", "#", '\1' }
};

TypeDetRulePrelim gVerbRulesPrelim[]= {
  { '\0', "j", "#", '\0' },
  { '\1', "t", "", '\0'  },
  { '\1', "WWt", "34_V", '\1' },
  { '\1', "Wst", "34_V", '\1' },
  { '\1', "!WWt", "35_V", '\1' },
  { '\1', "d", "", '\0'  },
  { '\1', "WUd", "34_V", '\1' },
  { '\1', "!WWnd", "34_V", '\1' },
  { '\1', "s", "", '\0'  },
  { '\1', "WWs", "32_V", '\1' },
  { '\1', "WKs", "32_V", '\1' },
  { '\1', "WWN", "33_V", '\0' },
  { '\1', "C", "#", '\0' },
  { '\2', "a", "", '\0'  },
  { '\2', "WWta", "27_V", '\1' },
  { '\2', "WWsta", "27_V", '\1' },
  { '\2', "WWNda", "27_V", '\1' },
  { '\2', "!WWCa", "27_V", '\1' },
  { '\2', "!WWGJa", "27_V", '\1' },
  { '\2', "Ta", "29_V", '\1' },
  { '\2', "Wia", "29_V", '\1' },
  { '\2', "e", "", '\0'  },
  { '\2', "Cle", "30_V", '\1' },
  { '\2', "Te", "27_V", '\1' },
  { '\2', "i", "", '\0'  },
  { '\2', "!WWZi", "27_V", '\1' },
  { '\2', "!WWGJi", "27_V", '\1' },
  { '\2', "WWbi", "27_V", '\1' },
  { '\2', "Ti", "28_V", '\1' },
  { '\2', "Fi", "28_V", '\1' },
  { '\2', "u", "", '\0'  },
  { '\2', "WKsu", "28_V", '\1' },
  { '\2', "LKsu", "28_V", '\1' },
  { '\2', "WWku", "28_V", '\1' },
  { '\2', "WWlu", "28_V", '\1' },
  { '\2', "WWnu", "28_V", '\1' },
  { '\2', "Wlku", "28_V", '\1' },
  { '\2', "Wnku", "28_V", '\1' },
  { '\2', "Tu", "27_V", '\1' },
  { '\2', "Wiu", "27_V", '\1' },
  { '\2', "W", "#", '\0' },
  { '\3', "a", "", '\0'  },
  { '\3', "Ata", "27_V", '\1' },
  { '\3', "Sta", "27_V", '\1' },
  { '\3', "Nda", "27_V", '\1' },
  { '\3', "vda", "27_V", '\1' },
  { '\3', "e", "", '\0'  },
  { '\3', "ise", "27_V", '\1' },
  { '\3', "tse", "27_V", '\1' },
  { '\3', "Ane", "27_V", '\1' },
  { '\3', "vne", "27_V", '\1' },
  { '\3', "ele", "31_V~27_V", '\1' },
  { '\3', "le", "27_V", '\1' },
  { '\3', "i", "", '\0'  },
  { '\3', "WWCi", "28_V", '\1' },
  { '\3', "TTi", "28_V", '\1' },
  { '\3', "QQi", "28_V", '\1' },
  { '\3', "u", "", '\0'  },
  { '\3', "stu", "27_V", '\1' },
  { '\3', "Ndu", "27_V", '\1' },
  { '\3', "Atu", "27_V", '\1' },
  { '\3', "eeru", "27_V", '\1' },
  { '\3', "Tu", "27_V", '\1' },
  { '\3', "W", "#", '\0' },
  { '\4', "a", "", '\0'  },
  { '\4', "Sta", "27_V", '\1' },
  { '\4', "Ata", "27_V", '\1' },
  { '\4', "Nda", "27_V", '\1' },
  { '\4', "e", "", '\0'  },
  { '\4', "ele", "36_V", '\1' },
  { '\4', "tse", "27_V", '\1' },
  { '\4', "i", "", '\0'  },
  { '\4', "WWCi", "28_V", '\1' },
  { '\4', "TTi", "28_V", '\1' },
  { '\4', "u", "", '\0'  },
  { '\4', "eeru", "27_V", '\1' },
  { '\4', "stu", "27_V", '\1' },
  { '\4', "Tu", "27_V", '\1' },
  { '\4', "W", "#", '\0' }
};

TypeDetRulePrelim gPosRulesPrelim[]= {
  { '\0', "j", "#", '\0' },
  { '\1', "W", "N", '\0' },
  { '\1', "C", "N", '\0' },
  { '\2', "Wma", "N", '\0' },
  { '\2', "Cma", "", '\0'  },
  { '\2', "tma", "V", '\1' },
  { '\2', "dma", "V", '\1' },
  { '\2', "WWsma", "V", '\1' },
  { '\2', "WKsma", "V", '\1' },
  { '\2', "WWNma", "V", '\1' },
  { '\2', "Cma", "N", '\1' },
  { '\2', "W", "N", '\0' },
  { '\2', "C", "N", '\0' },
  { '\3', "Cma", "N", '\0' },
  { '\3', "Wma", "", '\0'  },
  { '\3', "W1W1ma", "N", '\1' },
  { '\3', "oma", "N", '\1' },
  { '\3', "Ama", "V", '\1' },
  { '\3', "W", "N", '\0' },
  { '\3', "C", "N", '\0' },
  { '\4', "Cma", "N", '\0' },
  { '\4', "Wma", "", '\0'  },
  { '\4', "W1W1ma", "N", '\1' },
  { '\4', "oma", "N", '\1' },
  { '\4', "Ama", "V", '\1' },
  { '\4', "W", "N", '\0' },
  { '\4', "C", "N", '\0' }
};

/***************************************************************************/

const int kMaxSyllables = 5;

static TypeDetRulesList gNomenRules[kMaxSyllables];
static TypeDetRulesList gVerbRules[kMaxSyllables];
static TypeDetRulesList gPOSRules[kMaxSyllables];
static TypeDetRulesList gStemRules[kMaxSyllables];

static IndeclExcRegistry gTypeDetExcRegistry("typedet.exc");
static IndeclExcRegistry gStemDetExcRegistry("stemdet.exc");


int TypeDetRule::numberOfSigns() const
{
   int nrOfSigns=0;
   int patLength=strlen(fPattern);
   for(int i=0; i<patLength; i++)
      if(fPattern[i]=='!' || fPattern[i]=='1' || fPattern[i]=='2')
         nrOfSigns++;
   return nrOfSigns;
}

TypeDetRule::TypeDetRule()
{
  *fPattern='\0';
  fCompLength=0;
}

deleteRulesList(TypeDetRulesList &rulesList)
{
  TypeDetRulesList::iterator it;
  while(!rulesList.empty()){
    TypeDetRule *pTmp=rulesList.front();
    rulesList.pop_front();
    delete pTmp;
  }
}

TypeDetRule::~TypeDetRule()
{
  if(!fSubRules.empty())
    deleteRulesList(fSubRules);
}


TypeDetRule::TypeDetRule(const char *pat, const char *res) :
fResult(res? res : "")
{
  strcpy(fPattern, pat);
  fCompLength=strlen(fPattern)-numberOfSigns();
}


void TypeDetRule::addSubRule(TypeDetRule *pSub)
{
  fSubRules.push_back(pSub);
}

const char* testList(TypeDetRulesList &rulesList, const char *aWord)
{
  const char *result=0;
  TypeDetRulesList::iterator it;
  for(it=rulesList.begin(); it!=rulesList.end(); it++) {
    if(result=(*it)->test(aWord))
      break;
  }
  return result;
}

const char* TypeDetRule::test(const char *aWord) const
{
  int matchPos=strlen(aWord)-fCompLength;
  if(matchPos<0) {
    return 0;
  }
  if(matchesFromEnd(aWord, fPattern, matchPos+1)){
    if(!fSubRules.empty()){
      return testList(fSubRules, aWord);
    }
    else
      return fResult.c_str();
  }
  else {
    return 0;
  }
}

int loadTypeDetRulesFromFile(char *fileName, TypeDetRulesList *ruleArray)
{
  FILE *ruleFile=openAtLoc(fileName, "r", false);
  if(!ruleFile)
    return 0;
  char buf[512];
  int curNumSyll=0;
  TypeDetRule *pCurMainRule=0;
  while(fgets(buf, 512, ruleFile)){
    char *marker=strtok(buf, " \t\n");
    if(!marker)
      continue;
    switch(*marker){
    case ';': //comment
      //do nothing
      break;
    case '$': //syllables #
      {
        char *numSyllStr=strtok(0, " \t\n");
        if(!strcmp(numSyllStr, "1+"))
          curNumSyll=0;
        else
          curNumSyll=*numSyllStr-'0';
      }
      break;
    case '@': //main pattern
      {
        char *patStr=strtok(0, " \t\n");
        char *resStr=strtok(0, " \t\n");
        ruleArray[curNumSyll].push_back(new TypeDetRule(patStr, resStr));
        pCurMainRule=ruleArray[curNumSyll].back();
      }
      break;
    default:
      {
        if(pCurMainRule){
          char *patStr=marker;
          char *resStr=strtok(0, " \t\n");
          pCurMainRule->addSubRule(new TypeDetRule(patStr, resStr));
        }
      }
      break;
    }
  }

  fclose(ruleFile);
  return 1;
}

void loadTypeDetRulesFromArray(TypeDetRulePrelim *prelimArray, int numItems,
			       TypeDetRulesList *ruleArray)
{
  if(!prelimArray)
    return;

  TypeDetRule *pCurMainRule=0;

  for(int i=0; i<numItems; i++) {
    if(!prelimArray[i].fSub) {
      pCurMainRule=new TypeDetRule(prelimArray[i].fPattern,
				   prelimArray[i].fResult);
      ruleArray[prelimArray[i].fNumSyll].push_back(pCurMainRule);
    }
    else if(pCurMainRule)
      pCurMainRule->
	addSubRule(new TypeDetRule(prelimArray[i].fPattern,
				   prelimArray[i].fResult));
  }
}

void loadTypeDetRules(char *fileName,
		      TypeDetRulePrelim *prelimArray, int numItems,
		      TypeDetRulesList *ruleArray)
{
  if(!loadTypeDetRulesFromFile(fileName, ruleArray))
    loadTypeDetRulesFromArray(prelimArray, numItems, ruleArray);
}


int getMaxNumSyllables(char *inputBuf)
{
	int nMaxSyll=0; //default 0 means a syllabification error
  SyllabificationResult *pFirstSyllRes=(SyllabificationResult*)syllabify(inputBuf);
  if(pFirstSyllRes){
    //find maximum number of syllables
    nMaxSyll=pFirstSyllRes->numSyllables;
    for(SyllabificationResult *pCurSyllRes=pFirstSyllRes->next;  pCurSyllRes;
        pCurSyllRes=pCurSyllRes->next)
      if(pCurSyllRes->numSyllables>nMaxSyll)
        nMaxSyll=pCurSyllRes->numSyllables;
      deleteResults(pFirstSyllRes);
   }
   return nMaxSyll;
}

int isVerb(char *inputBuf, int nMaxSyll)
{
  //there are common rulesets for >= 4 syllables:
  if(nMaxSyll > 4)
    nMaxSyll = 4;
  //try "universal" rules
  const char *result=testList(gPOSRules[0], inputBuf);
  if(!result)
    //try rules corresponding to the number of syllables
    result=testList(gPOSRules[nMaxSyll], inputBuf);

  if(result && *result=='V')
    return 1;
  else
    return 0;
}

void findTypesByRules(char *inputBuf, char *outputBuf, TypeDetRulesList *ruleArray,
                      int nMaxSyll)
{
  //there are common rulesets for >= 4 syllables:
  if(nMaxSyll > 4)
    nMaxSyll = 4;
  //try "universal" rules
  const char *result=testList(ruleArray[0], inputBuf);
  if(!result)
    //try rules corresponding to the number of syllables
    result=testList(ruleArray[nMaxSyll], inputBuf);

  if(result){
    if(*outputBuf)
      strcat(outputBuf, "|");
    strcat(outputBuf, result);
  }
}


extern "C" int _stdcall FindTypes(char *inputString, char *outputString,
                                  int outputLength,
                                  int withAppostr)
{
  char buf[256]; //output
  *buf=0;
  int rulesToo=0;
  int n=gTypeDetExcRegistry.findExceptions(buf, inputString, withAppostr, &rulesToo);
  if(!n || n && rulesToo){

    //silbitus ei taha va"ltema"rke!
    char syllInput[256];
    strcpy(syllInput, inputString);
    if(withAppostr)
      deAppostrophise(syllInput);

    int nSyll=getMaxNumSyllables(syllInput);

    if(nSyll>0){
      char inputBuf[256];
      //put # to the beginning of the input word
      //this trick enables to determine words like 'helama' & 'elama'
      //by the same rules
      strcpy(inputBuf, "#");
      strcat(inputBuf, inputString);
      int l=strlen(inputBuf);

      if(isVerb(inputBuf, nSyll)){
        if(l>2){
          //remove 'ma'
          inputBuf[l-2]='\0';
          //and decrement the number of syllables
          nSyll--;
        }
        findTypesByRules(inputBuf, buf, gVerbRules, nSyll);
      }
      else{
        findTypesByRules(inputBuf, buf, gNomenRules, nSyll);
      }
    }
  }

  strncpy(outputString, buf, outputLength);
  return 0;
}



extern "C" int _stdcall FindStemTypes(char *inputString, char *outputString,
                                      int outputLength,
                                      int withAppostr)
{
  char buf[512]; //output
  *buf=0;
  int rulesToo=0;
  int n=gStemDetExcRegistry.findExceptions(buf, inputString, withAppostr, &rulesToo);
  if(!n || n && rulesToo){

    //silbitus ei taha va"ltema"rke!
    char syllInput[256];
    strcpy(syllInput, inputString);
    if(withAppostr)
      deAppostrophise(syllInput);

    int nSyll=getMaxNumSyllables(syllInput);
    if(nSyll>0) {
    char inputCopy[256];
      //put # to the beginning of the input word
      //this trick enables to determine words like 'helama' & 'elama'
      //by the same rules
      strcpy(inputCopy, "#");
      strcat(inputCopy, inputString);

      findTypesByRules(inputCopy, buf, gStemRules, nSyll);
    }
  }

  strncpy(outputString, buf, outputLength);
  return 0;
}



#ifdef LINUX_PORT
extern "C" void initTypeDetector()
{
  initPatternMatcher();
  initSyllabification();
  gTypeDetExcRegistry.createIndices();
  loadTypeDetRules("noun.rul", gNounRulesPrelim,
		   sizeof(gNounRulesPrelim)/sizeof(TypeDetRulePrelim),
		   gNomenRules);
  loadTypeDetRules("verb.rul",
		   gVerbRulesPrelim,
		   sizeof(gVerbRulesPrelim)/sizeof(TypeDetRulePrelim),
		   gVerbRules);
  loadTypeDetRules("pos.rul",
		   gPosRulesPrelim,
		   sizeof(gPosRulesPrelim)/sizeof(TypeDetRulePrelim),
		   gPOSRules);
#if 0
  gStemDetExcRegistry.createIndices();
  loadTypeDetRules("stemdet.rul",
       NULL,
       sizeof(gStemRulesPrelim)/sizeof(TypeDetRulePrelim),
       gStemRules);
#endif
}

extern "C" void uninitTypeDetector()
{
  for(int i=0; i<kMaxSyllables; i++){
    deleteRulesList(gNomenRules[i]);
    deleteRulesList(gVerbRules[i]);
    deleteRulesList(gPOSRules[i]);
#if 0
    deleteRulesList(gStemRules[i]);
#endif
  }
}


#else

BOOL WINAPI DllEntryPoint(HANDLE hinstDll, DWORD fdwReason, LPVOID lpvReserved)
{
  if(fdwReason==DLL_PROCESS_ATTACH){
    gTypeDetExcRegistry.createIndices();
    loadTypeDetRules("noun.rul", gNounRulesPrelim,
		     sizeof(gNounRulesPrelim)/sizeof(TypeDetRulePrelim),
		     gNomenRules);
    loadTypeDetRules("verb.rul",
		     gVerbRulesPrelim,
		     sizeof(gVerbRulesPrelim)/sizeof(TypeDetRulePrelim),
		     gVerbRules);
    loadTypeDetRules("pos.rul",
		     gPosRulesPrelim,
		     sizeof(gPosRulesPrelim)/sizeof(TypeDetRulePrelim),
		     gPOSRules);

    gStemDetExcRegistry.createIndices();
    loadTypeDetRules("stemdet.rul",
		     NULL,
		     0,
		     gStemRules);
  }
  else if(fdwReason==DLL_PROCESS_DETACH){
    for(int i=0; i<kMaxSyllables; i++){
      deleteRulesList(gNomenRules[i]);
      deleteRulesList(gVerbRules[i]);
      deleteRulesList(gPOSRules[i]);

      deleteRulesList(gStemRules[i]);
    }
  }

   return 1;
}

#endif














