#include "grammar.hpp"

#include <stdio.h>
#include <ctype.h>

#ifdef LINUX_PORT
#define _stdcall
#endif

#include <openfile.h>
#include "patmatch_exp.h"


int numberOfSigns(const string &pattern)
{
   int nrOfSigns=0;
   for(int i=0; i<pattern.length(); i++)
      if(pattern[i]=='!' || pattern[i]=='1' || pattern[i]=='2')
         nrOfSigns++;
   return nrOfSigns;
}


Rule::Rule(string upper, string lower, string leftContext, string rightContext)
{
   fUpper=upper;
   fLower=lower;
   fLeftContext=leftContext;
   fRightContext=rightContext;

   fWholePattern=fLeftContext+fUpper+fRightContext;
   fCompLength=fWholePattern.length()-numberOfSigns(fWholePattern);

   fLeftLength = fLeftContext.length() - numberOfSigns(fLeftContext);
}

Rule::Rule():fUpper(), fLower(), fLeftContext(), fRightContext(), fCompLength(0),
fLeftLength(0)
{
}


Rule::Rule(const Rule &rule)
{
   fUpper=rule.fUpper;
   fLower=rule.fLower;
   fLeftContext=rule.fLeftContext;
   fRightContext=rule.fRightContext;
   fCompLength=rule.fCompLength;
   fWholePattern=rule.fWholePattern;
   fLeftLength=rule.fLeftLength;
}

Rule::~Rule()
{
}

Rule& Rule::operator=(const Rule &rule)
{
   fUpper=rule.fUpper;
   fLower=rule.fLower;
   fLeftContext=rule.fLeftContext;
   fRightContext=rule.fRightContext;
   fCompLength=rule.fCompLength;
   fWholePattern=rule.fWholePattern;
   fLeftLength=rule.fLeftLength;
   return *this;
}

int Rule::operator==(const Rule &rule)
{
   return    fUpper==rule.fUpper &&
             fLower==rule.fLower &&
             fLeftContext==rule.fLeftContext &&
             fRightContext==rule.fRightContext;
}

int Rule::isMatching(const string &inString, int posToCompare, int *ppos)
{
  int result=matchesFromEnd(inString.c_str(), fWholePattern.c_str(), posToCompare+1);
  if(result){
    *ppos=result+fLeftLength-1;
    return 1;
  }
  else
    return 0;

}

string Rule::transform(const string &inString, int pos)
{
   string outString=inString;

   string newPart(fLower.length()-numberOfSigns(fLower), 'x');
   int k=0;
   for(int i=0; i<fLower.length(); i++)
      if(fLower[i]!='!' && fLower[i]!='1' && fLower[i]!='2')
      {
         if(isupper(fLower[i]))
         {
            if(i<fUpper.length() && fUpper[i]==fLower[i])
               newPart[k]=inString[pos+k];
            else if(i>0 && fUpper[i-1]==fLower[i] || i>1 && fUpper[i-2]==fLower[i])
               newPart[k]=inString[pos+k-1];
            else
               newPart[k]=fLower[i];
         }
         else
            newPart[k]=fLower[i];
         k++;
      }

   outString.remove(pos, fUpper.length()-numberOfSigns(fUpper));
   outString.insert(pos, newPart);
   return outString;
}

int Rule::lengthOfComparablePart()
{
   return fCompLength;
}

string Rule::asText()
{
   return  fUpper+" "+fLower+" "+fLeftContext+" "+fRightContext;
}


Grammar::Grammar(string grammarName):
fRulesWithAppostr(),
fRulesWithoutAppostr(),
flags(0), fpExcRegistry(0), fBaseIsLeft(0), fName(grammarName)
{
}

Grammar::~Grammar()
{
}

int Grammar::loadFromFile(const char * const fileName)
{
   FILE *ruleFile=openAtLoc(fileName, "r", false);
   if(!ruleFile)
      return 0; //failure!
   char upperBuf[10], lowerBuf[10], leftCtxBuf[10], rightCtxBuf[10], lineBuf[80];
   while(fgets(lineBuf, 80, ruleFile))
   {
      switch(*lineBuf)
      {
      case '%':
         if(!strncmp(lineBuf+1, "+v", 2)){
            flags|=kDegree;
            flags|=kPlus;
         }
         else if(!strncmp(lineBuf+1, "-v", 2)){
            flags|=kDegree;
            flags&=~kPlus;
         }
         else if(!strncmp(lineBuf+1, "p0", 2))
            flags|=kZeroPos;
         else if(!strncmp(lineBuf+1, "ori", 3))
            flags|=kOrigOutput;
         break;
      case '/':
         /* Comment! */
         break;
      default:
         sscanf(lineBuf, "%s %s %s %s",
                upperBuf, lowerBuf, leftCtxBuf, rightCtxBuf);
         *upperBuf=*upperBuf=='0'? '\0': *upperBuf;
         *lowerBuf=*lowerBuf=='0'? '\0': *lowerBuf;
         *leftCtxBuf=*leftCtxBuf=='0'? '\0': *leftCtxBuf;
         *rightCtxBuf=*rightCtxBuf=='0'? '\0': *rightCtxBuf;
         fRulesWithAppostr.push_back(Rule(upperBuf, lowerBuf, leftCtxBuf, rightCtxBuf));
         fRulesWithoutAppostr.push_back(Rule(removeAppostrophes(upperBuf),
                                        removeAppostrophes(lowerBuf),
                                        removeAppostrophes(leftCtxBuf),
                                        removeAppostrophes(rightCtxBuf)));
         break;
      }
   }
   fclose(ruleFile);

   return 1; //success!
}


void Grammar::loadFromArray(RulePrelim *pStaticRules, int nrStaticRules)
{
   char upperBuf[10], lowerBuf[10], leftCtxBuf[10], rightCtxBuf[10];
   for(int i=0; i<nrStaticRules; i++)
   {
      strcpy(upperBuf, (pStaticRules+i)->left);
      strcpy(lowerBuf, (pStaticRules+i)->right);
      strcpy(leftCtxBuf, (pStaticRules+i)->leftCtx);
      strcpy(rightCtxBuf, (pStaticRules+i)->rightCtx);
      *upperBuf=*upperBuf=='0'? '\0': *upperBuf;
      *lowerBuf=*lowerBuf=='0'? '\0': *lowerBuf;
      *leftCtxBuf=*leftCtxBuf=='0'? '\0': *leftCtxBuf;
      *rightCtxBuf=*rightCtxBuf=='0'? '\0': *rightCtxBuf;
      fRulesWithAppostr.push_back(Rule(upperBuf, lowerBuf, leftCtxBuf, rightCtxBuf));
      fRulesWithoutAppostr.push_back(Rule(removeAppostrophes(upperBuf),
                                     removeAppostrophes(lowerBuf),
                                     removeAppostrophes(leftCtxBuf),
                                     removeAppostrophes(rightCtxBuf)));
   }
}



Grammar& Grammar::addRule(const Rule &rule)
{
   return *this;
}


void processBrackets(ResultCollection *pResult, DerivationResult *pOutput,
                     DerivationResult *pInput)
{
   string inString=pOutput->value();
   int pos1=inString.find("(");
   if(pos1!=string::npos)
   {
      int pos2=inString.find(")", pos1+1);
      if(pos2!=string::npos && pos2>pos1)
      {
         string comp1=inString;
         comp1.remove(pos1, pos2-pos1+1);
         string comp2=inString;
         comp2.remove(pos2, 1);
         comp2.remove(pos1, 1);
         pOutput->setValue(comp1);
         DerivationResult *pOther=pInput->newResult(comp2);
         pResult->push_back(pOther);
      }
   }
}

void Grammar::transform(ResultCollection *pResult, DerivationResult *pInput,
                        int withAppostr, const string &tokenFromExceptions, int *msg)
{
   string corrString="#"+pInput->value()+"#";
   int position;
   int compLength=0;
   if(flags & kDegree)
      compLength=significantPosition(pInput->value());
   else if( flags & kZeroPos)
      compLength=0;
   RuleArray *pRuleArray=withAppostr? &fRulesWithAppostr: &fRulesWithoutAppostr;
   RuleArray::iterator iter=pRuleArray->begin();
   while(iter!=pRuleArray->end())
   {
      if(!(flags & kDegree) && !(flags & kZeroPos))
         compLength=corrString.length()-(*iter).lengthOfComparablePart();
      if(compLength>=0 && (*iter).isMatching(corrString, compLength, &position))
         break;

      iter++;
   }
   if(iter!=pRuleArray->end() || (flags & kOrigOutput))
   {
      if(iter!=pRuleArray->end())
         corrString=(*iter).transform(corrString, position);

      if(corrString.length()>0 && corrString[0]=='#')
         corrString.remove(0, 1);
      if(corrString.length()>0 && corrString[corrString.length()-1]=='#')
         corrString=corrString.remove(corrString.length()-1, 1);

      if(withAppostr)
      {
         if(flags & kDegree)
         {
            int pos=significantPosition(corrString);
            if((flags & kPlus) && (!pos || pos && corrString[pos-1]!='\''))
               corrString.insert(pos, "'");
            else if(!(flags & kPlus) && (pos>0 && corrString[pos-1]=='\''))
               corrString.remove(pos-1, 1);
         }
      }

      DerivationResult *pOutput=pInput->newResult(corrString, "", tokenFromExceptions);
      pResult->push_back(pOutput);
      processBrackets(pResult, pOutput, pInput);
   }
   else
   {
      DerivationResult *pOutput=pInput->newResult("0", "", tokenFromExceptions);
      pOutput->setIgnored();
      pResult->push_back(pOutput);
      if(msg)
         *msg=1;
   }
}


Grammar& Grammar::unload()
{
   return *this;
}


Grammar& Grammar::setExceptionRegistry(ExceptionRegistry *pExcRegistry,
                                       int baseIsLeft)
{
   fpExcRegistry=pExcRegistry;
   fBaseIsLeft=baseIsLeft;
   return *this;
}

int Grammar::findExceptionalForms(ResultCollection *pResult,
                                  DerivationResult *pInput,
                                  int typeId, int withAppostr,
                                  int *pHasRuleBasedAlso,
                                  string &tokenFromExceptions)
{
   if(fpExcRegistry)
      return fpExcRegistry->findExceptions(pResult, pInput, flip(fBaseIsLeft), typeId,
                                           fName, withAppostr, pHasRuleBasedAlso,
                                           tokenFromExceptions);
   else
      return 0;
}


unsigned int firstWowelOfSyllable(const string &aWord, int l)
{
   //pass trailing non-wowels
   while(l>=0 && !isCharOfClass(aWord[l], 'W'))
      l--;
   //pass wowels
   while(l>=0 && isCharOfClass(aWord[l], 'W'))
      l--;
   return l+1;
}


unsigned int significantPosition(const string &aWord)
{
   int z=aWord.length()-1;
   //find position of the last wowel or consonant of the string
   while (z>=0 && !isCharOfClass(aWord[z], 'W')
               && !isCharOfClass(aWord[z], 'C'))
      z--;
   //find position of the first wowel of the last syllable
   int pos=firstWowelOfSyllable(aWord, z);

   //if the ending pattern is CW# or CWC#
   if(isCharOfClass(aWord[z], 'W') &&
      z>=1 && isCharOfClass(aWord[z-1], 'C') ||
      isCharOfClass(aWord[z], 'C') &&
      z>=2 && isCharOfClass(aWord[z-1], 'W') &&
      isCharOfClass(aWord[z-2], 'C'))
      pos=firstWowelOfSyllable(aWord, pos-1);

   return pos;
}
