
#ifdef LINUX_PORT
#define _stdcall
#else
#include <windows.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <set>

#include <openfile.h>

#ifdef __BORLANDC__
typedef std::set<char, std::less<char> > CharSet;
#else
typedef set<char, less<char> > CharSet;
#endif

const int kNoConstraint=0;
const int kNegation=1;
const int kEqual=2;
const int kNotEqual=3;

class CharClassInfo
{
public:
   CharClassInfo();
   ~CharClassInfo();
   CharClassInfo& loadFromFile(const char * const fileName);
   int isA(char c, char aClass);
   char basicClass(char c);
private:
   CharSet fClassesDict[256];
   char fAllowedChars[256];
};

CharClassInfo CharClassInfoObject;


CharClassInfo::CharClassInfo()
{
  memset(fAllowedChars, '\0', 256);
}

CharClassInfo::~CharClassInfo()
{
}

CharClassInfo& CharClassInfo::loadFromFile(const char * const fileName)
{
  FILE *charsetFile=openAtLoc(fileName, "r");
  if(!charsetFile)
    throw "Error on opening charset file!";
  char buf[256];
  while(fgets(buf, 256, charsetFile)){
    char *pClass=strtok(buf, " \t\n");
    if(pClass){
      char *pChar;
      while(pChar=strtok(0, " \t\n")){
        fClassesDict[(unsigned char)*pChar].insert(*pClass);
        if(*pClass=='W' || *pClass=='C')
          fAllowedChars[(unsigned char)*pChar]=*pClass;
      }
    }
  }
  fclose(charsetFile);

  return *this;
}


inline int CharClassInfo::isA(char c, char aClass)
{
  unsigned char uc=(unsigned char)c;
  return fClassesDict[uc].find(aClass)!=fClassesDict[uc].end();
}

inline char CharClassInfo::basicClass(char c)
{
  return fAllowedChars[(unsigned char)c];
}

inline int numberOfSigns(char *pattern)
{
   int nrOfSigns=0;
   int patLength=strlen(pattern);
   for(int i=0; i<patLength; i++)
      if(pattern[i]=='!' || pattern[i]=='1' || pattern[i]=='2')
         nrOfSigns++;
   return nrOfSigns;
}


inline int constraint(char *pattern, int pos)
{
  int constr=kNoConstraint;
  int patLength=strlen(pattern);
  if(pos>0)
  {
    if(pattern[pos-1]=='!')
      constr=kNegation;
    else if(pos>1 && pos<patLength-1)
    {
      if(pattern[pos]==pattern[pos-2])
      {
        if(pattern[pos-1]=='1' && pattern[pos+1]=='1' ||
           pattern[pos-1]=='2' && pattern[pos+1]=='2')
          constr=kEqual;
        else if(pattern[pos-1]=='1' && pattern[pos+1]=='2' ||
                pattern[pos-1]=='2' && pattern[pos+1]=='1')
          constr=kNotEqual;
      }
    }
  }
  return constr;
}

inline int isMatchingAt(char *word, char *pattern, int patLength, int i)
{
  int k=0, misMatch=0;
  for(int j=0; j<patLength && !misMatch; j++)
    if(pattern[j]!='!' && pattern[j]!='1' && pattern[j]!='2'){
      misMatch=1;
      if(isupper(pattern[j])){
        int constr=constraint(pattern, j);
        switch(constr)
        {
        case kNegation:
          if(!CharClassInfoObject.isA(word[i+k], pattern[j]))
            misMatch=0;
          break;
        case kEqual:
          if(CharClassInfoObject.isA(word[i+k], pattern[j]) &&
            word[i+k]==word[i+k-1])
            misMatch=0;
          break;
        case kNotEqual:
          if(CharClassInfoObject.isA(word[i+k], pattern[j]) &&
            word[i+k]!=word[i+k-1])
              misMatch=0;
            break;
        default:
          if(CharClassInfoObject.isA(word[i+k], pattern[j]))
            misMatch=0;
          break;
        }
      }
      else if(pattern[j]==word[i+k])
        misMatch=0;
      k++;
    }

  return !misMatch;
}

extern "C" int _stdcall matchesFromEnd(char *word, char *pattern,
                                       int posToCompare)
{
  int wordLength=strlen(word);
  int patLength=strlen(pattern);
  int patCharLength=patLength-numberOfSigns(pattern);

  for(int i=wordLength-patCharLength; i>=posToCompare-1; i--)
    if(isMatchingAt(word, pattern, patLength, i))
      return i+1;

  return 0;
}

extern "C" int _stdcall matchesFromStart(char *word, char *pattern,
                                         int posToCompare)
{
  int wordLength=strlen(word);
  int patLength=strlen(pattern);
  int patCharLength=patLength-numberOfSigns(pattern);

  for(int i=posToCompare-1; i<=wordLength-patCharLength; i++)
    if(isMatchingAt(word, pattern, patLength, i))
      return i+1;

  return 0;
}

extern "C" int _stdcall isCharOfClass(char c, char aClass)
{
  return CharClassInfoObject.isA(c, aClass);
}

extern "C" int _stdcall isEstonianWord(char *pWord)
{
  while(*pWord){
    if(!CharClassInfoObject.basicClass(*pWord))
      return 0;
    pWord++;
  }
  return 1;
}

#ifdef LINUX_PORT
extern "C" void initPatternMatcher()
{
  static int patMatchInitialised = 0;
  if(!patMatchInitialised) {
    CharClassInfoObject.loadFromFile("chars.ini");
    patMatchInitialised = 1;
  }
}
#else
BOOL WINAPI DllEntryPoint(HANDLE hinstDll, DWORD fdwReason, LPVOID lpvReserved)
{
	if(fdwReason==DLL_PROCESS_ATTACH){
   	CharClassInfoObject.loadFromFile("chars.ini");
  }

  return 1;
}
#endif





