#ifndef LINUX_PORT
#include <mem.h>
#endif

#include <stdio.h>

#include "stemmgr.hpp"
#include <openfile.h>

#ifdef __BORLANDC__
typedef std::pair<const std_string, Grammar*> StringToGrammarAssoc;
typedef std::deque<StemNode*> NodeQueue;
#else
typedef pair<const std_string, Grammar*> StringToGrammarAssoc;
typedef deque<StemNode*> NodeQueue;
#endif


StemNode::StemNode(const std_string &stemCode):fValue(stemCode),fpSibling(0),
fpFirstChild(0), fpParent(0), fOrtNr(kNoOrt), fExceptionalTransition(0),
fIntermediate(0)
{
}

StemNode::~StemNode()
{
   if(fpFirstChild)
   {
      delete fpFirstChild;
      fpFirstChild=0;
   }
   if(fpSibling)
   {
      delete fpSibling;
      fpSibling=0;
   }
}


std_string StemNode::value() const
{
   if(fIntermediate)
      return "("+fValue+")";
   else
      return fValue;
}

std_string StemNode::pureValue() const
{
   return fValue;
}

StemNode* StemNode::createSibling(const std_string &stemCode, int ortNr,
                                  int bExceptionalTransition, int bIntermediate)
{
   fpSibling=new StemNode(stemCode);
   fpSibling->fpParent=fpParent;
   fpSibling->fExceptionalTransition=bExceptionalTransition;
   fpSibling->fOrtNr=ortNr;
   fpSibling->fIntermediate=bIntermediate;
   return fpSibling;
}

StemNode* StemNode::createChild(const std_string &stemCode, int ortNr,
                                int bExceptionalTransition, int bIntermediate)
{
   if(!fpFirstChild)
   {
      fpFirstChild=new StemNode(stemCode);
      fpFirstChild->fpParent=this;
      fpFirstChild->fExceptionalTransition=bExceptionalTransition;
      fpFirstChild->fOrtNr=ortNr;
      fpFirstChild->fIntermediate=bIntermediate;
      return fpFirstChild;
   }
   else
   {
      StemNode *pNode;
      for(pNode=fpFirstChild; pNode->fpSibling; pNode=pNode->fpSibling);
      return pNode->createSibling(stemCode, ortNr, bExceptionalTransition, bIntermediate);
   }
}

StemNode* StemNode::findStemNode(const std_string &stemCode)
{
   StemNode *pNode=0;
   if(stemCode==fValue)
      pNode=this;
   else
   {
      if(fpFirstChild)
         pNode=fpFirstChild->findStemNode(stemCode);
      if(!pNode && fpSibling)
         pNode=fpSibling->findStemNode(stemCode);
   }
   return pNode;
}



StemManager::StemManager()
{
   for(int k=0; k<kNumberOfOrtographies; k++)
      fOrtography[k]="";
   memset(firstStem, 0, sizeof(firstStem));
}

StemManager::~StemManager()
{
   for(StringToGrammarDict::iterator iter=fGrammarDict.begin();
       iter!=fGrammarDict.end(); iter++)
      delete (*iter).second;

   for(int i=0; i<kNumberOfStemTypes; i++)
   {
      if(firstStem[i])
      {
         delete firstStem[i];
         firstStem[i]=0;
      }
   }
}

inline void readCodeBuf(char *pBuf, char c, FILE *f)
{
   *pBuf=c;
   int i=1;
   while(isalnum(c=fgetc(f)))
      *(pBuf+i++)=c;
   ungetc(c, f);
   *(pBuf+i)=0;
}

StemManager& StemManager::loadFromFile(const char * const fileName)
{
   FILE *stemCodeFile;
   if(!(stemCodeFile=openAtLoc(fileName, "r")))
      throw "Error on opening stem code file!";
   char c, c2, buf[4];
   int index=0;
   int state=0;
   int ortNr=kNoOrt, bExceptionalTransition=0, bExceptionalType=0, bIntermediate=0;
   StemNode *currentNode;
   while((c=fgetc(stemCodeFile))!=EOF)
   {
      switch(state)
      {
      case 0:  //index
         if(isdigit(c)){
            c2=fgetc(stemCodeFile);
            index=(c-'0')*10+(c2-'0');
            bExceptionalType=0;
            state=1;
         }
         break;
      case 1:  //after index
         if(isalpha(c)){
            readCodeBuf(buf, c, stemCodeFile);
            currentNode=firstStem[index]=new StemNode(buf);
            state=2;
         }
         else if(c=='#'){
            bExceptionalType=1;
         }
         break;
      case 2:  //after stem code
         switch(c)
         {
         case '>':
            bExceptionalTransition=0;
            ortNr=kNoOrt;
            bIntermediate=0;
            state=3;
            break;
         case ';':
            state=4;
            break;
         case '\n':
            state=0;
            break;
         }
         break;
      case 3:  //after '>'
         if(isalpha(c)){
            readCodeBuf(buf, c, stemCodeFile);
            currentNode=currentNode->
                        createChild(buf, ortNr, bExceptionalType || bExceptionalTransition,
                                    bIntermediate);
            state=2;
         }
         else if(c=='#'){
            bExceptionalTransition=1;
         }
         else if(c=='+'){
            ortNr=0;
         }
         else if(c=='*'){
            ortNr=1;
         }
         else if(c=='('){
            bIntermediate=1;
         }
         break;
      case 4:  //after ';'
         if(isalpha(c)){
            readCodeBuf(buf, c, stemCodeFile);
            currentNode=firstStem[index]->findStemNode(buf);
            state=2;
         }
         break;
      }
   }

   fclose(stemCodeFile);
   return *this;
}

const StemNode* StemManager::previousStemNode(const StemNode *pNode)
{
   return pNode->fpParent;
}

const StemNode* StemManager::getStemNode(int index, const std_string &stemCode)
{
   if(index>=0 && index<kNumberOfStemTypes && firstStem[index])
      return firstStem[index]->findStemNode(stemCode);
   else
      return 0;
}

StemManager& StemManager::addGrammar(const std_string &grammarName,
                                     const std_string &grammarFile,
                                     RulePrelim *pStaticRules, int nrStaticRules,
                                     ExceptionRegistry *pExcRegistry,
                                     int baseIsLeft)
{
   Grammar *pGrammar=new Grammar(grammarName);
#ifdef __BORLANDC__
   fGrammarDict.insert(std::make_pair(grammarName, pGrammar));
#else
   fGrammarDict.insert(StringToGrammarAssoc(grammarName, pGrammar));
#endif
   if(!pGrammar->loadFromFile(grammarFile.c_str())){
      pGrammar->loadFromArray(pStaticRules, nrStaticRules);
      if(grammarName=="ort1" || grammarName=="ort2")
         pGrammar->addFlags(Grammar::kOrigOutput);
      else if(grammarName=="0g"){
         pGrammar->addFlags(Grammar::kDegree);
         pGrammar->addFlags(Grammar::kPlus);
      }
      else if(grammarName=="g0"){
         pGrammar->addFlags(Grammar::kDegree);
         pGrammar->removeFlags(Grammar::kPlus);
      }
      else if(grammarName=="nt"){
         pGrammar->addFlags(Grammar::kOrigOutput);
         pGrammar->addFlags(Grammar::kDegree);
         pGrammar->addFlags(Grammar::kPlus);
      }
      else if(grammarName=="tn"){
         pGrammar->addFlags(Grammar::kOrigOutput);
         pGrammar->addFlags(Grammar::kDegree);
         pGrammar->removeFlags(Grammar::kPlus);
      }
   }
   pGrammar->setExceptionRegistry(pExcRegistry, baseIsLeft);
   return *this;
}

void StemManager::transform(ResultCollection *pResults, DerivationResult *pInput,
                            const std_string &grammarName, const std_string &targetStemCode,
                            int stemNr, int bExceptionalTransition, int withAppostr,
                            int *msg)
{
   if(pInput->isDeadEnd())
   {
      std_string newValue(pInput->value());
      if(pInput->value()!="0" && pInput->value()!="#")
         newValue="";
      DerivationResult *pOutput=pInput->newResult(newValue, targetStemCode);
      pOutput->setIgnored();
      pResults->push_back(pOutput);
      return;
   }
   const Grammar *pGrammar=findGrammar(grammarName);
   if(pGrammar)
   {
      int hasRuleBasedAlso=0, exceptionsFound=0;
      std_string tokenFromExceptions;
      exceptionsFound=pGrammar->findExceptionalForms(pResults, pInput, stemNr,
                                                     withAppostr, &hasRuleBasedAlso,
                                                     tokenFromExceptions);
      if(bExceptionalTransition)
      {
         if(!exceptionsFound)
         {
            DerivationResult *pOutput=pInput->newResult("#");
            pOutput->setIgnored();
            pResults->push_back(pOutput);
         }
      }
      else
      {
         if(!exceptionsFound || hasRuleBasedAlso)
            pGrammar->transform(pResults, pInput, withAppostr, tokenFromExceptions, msg);
      }
      ResultCollection::iterator iter;
      for(iter=pResults->begin(); iter!=pResults->end(); iter++)
         (*iter)->setStemCode(targetStemCode);
   }
   else
   {
      DerivationResult *pOutput=pInput->newResult("", targetStemCode);
      pOutput->setIgnored();
      pResults->push_back(pOutput);
   }

//    printf("%s input: %s\n", grammarName.c_str(), 
// 	  pInput->value().c_str());
   ResultCollection::iterator zzz=pResults->begin();
//    printf("output: %s\n", 
// 	  zzz!=pResults->end()?
// 	  (*zzz)->value().c_str():"NOTHING");
}


void StemManager::disAmbiguate(ResultCollection *pResults, DerivationResult *pInput,
                               const std_string &grammarName, const std_string &targetStemCode,
                               int stemNr, int bExceptionalTransition, int withAppostr, int *msg)
{
   if(!withAppostr && !bExceptionalTransition)
   {
      ResultCollection::iterator iter;
      for(iter=pResults->begin(); iter!=pResults->end(); iter++)
         if((*iter)->value()==pInput->value())
         {
            std_string reverseGrammar(grammarName);
            reverseGrammar[0]=grammarName[1];
            reverseGrammar[1]=grammarName[0];
            ResultCollection *pFict=new ResultCollection;
            DerivationResult *pToCheck=new DerivationResult(**iter);
            transform(pFict, pToCheck, reverseGrammar, targetStemCode, stemNr,
                      bExceptionalTransition, withAppostr, msg);
            int origFound=0;
            ResultCollection::iterator iter2;
            for(iter2=pFict->begin(); iter2!=pFict->end(); iter2++)
               if((*iter2)->value()==pInput->value())
                  origFound=1;
            if(!origFound)
               (*iter)->setValue("0");
            FlushCollection(*pFict);
            delete pFict;
         }
   }
}


StemManager& StemManager::unloadGrammars()
{
  StringToGrammarDict::iterator iter;
   for(iter=fGrammarDict.begin(); iter!=fGrammarDict.end(); iter++)
      delete (*iter).second;
   fGrammarDict.erase(fGrammarDict.begin(), fGrammarDict.end());

   return *this;
}

const Grammar* StemManager::findGrammar(const std_string &grammarName)
{
   StringToGrammarDict::iterator iter=fGrammarDict.find(grammarName);
   if(iter!=fGrammarDict.end())
      return (*iter).second;
   else
      return 0;
}


std_string grammarFromCodes(const std_string &code1, const std_string &code2)
{
   std_string grName("00");
   int i=0;
   while(i<code1.length() && i<code2.length() && code1[i]==code2[i])
      i++;
   grName[0]=i<code1.length()? code1[i]: '0';
   grName[1]=i<code2.length()? code2[i]: '0';
   return grName;
}


inline void resetCollections(ResultCollection **ppExisting,
                             ResultCollection **ppNew)
{
   ResultCollection *pTmp=*ppExisting;
   *ppExisting=*ppNew;
   pTmp->erase(pTmp->begin(), pTmp->end());
   *ppNew=pTmp;
}

void removeRepeatingParts(ResultCollection &coll)
{
   if(!coll.empty())
   {
      int n=coll.size(), i, j;
      for(i=0; i<n-1; i++)
         for(j=i+1; j<n; j++)
            if(coll[j]->value()==coll[i]->value())
               coll[j]->setRepeating();
   }
}


ResultCollection*  StemManager::createLemma(const std_string &inString, int stemNr,
                                            const std_string &stemCode, int withAppostr)
{
   ResultCollection *pExistingColl=new ResultCollection;
   const StemNode *pNode1, *pNode2=getStemNode(stemNr, stemCode);
   if(!pNode2)
   {
#if 0
      MessageBox(NULL, "Vale tp vi tvekood", "Viga!", MB_OK|MB_TASKMODAL);
#endif
      return pExistingColl;
   }

   ResultCollection *pNewColl=new ResultCollection;
   DerivationResult *pResult=new DerivationResult(inString, stemCode);
   pResult->enableHistory();
   pExistingColl->push_back(pResult);

   std_string grammarName("");
   do
   {
      pNode1=pNode2;
      pNode2=previousStemNode(pNode2);
      if(pNode2)
      {
         grammarName=grammarFromCodes(pNode1->pureValue(), pNode2->pureValue());
         ResultCollection::iterator iter;
         for(iter=pExistingColl->begin(); iter!=pExistingColl->end(); iter++){
           transform(pNewColl, *iter, grammarName, pNode2->value(),
                     stemNr, pNode1->hasExceptionalTransition(), withAppostr, 0);
            if(grammarName=="nt" || grammarName=="tn")
               disAmbiguate(pNewColl, *iter, grammarName, pNode2->value(),
                            stemNr, pNode1->hasExceptionalTransition(), withAppostr, 0);
         }
         resetCollections(&pExistingColl, &pNewColl);
         int ortNr=kNoOrt;
         if((ortNr=pNode1->ortography())!=kNoOrt)
         {
            ResultCollection::iterator iter;
            for(iter=pExistingColl->begin(); iter!=pExistingColl->end(); iter++)
              transform(pNewColl, *iter, fOrtography[ortNr], "ort",
                        stemNr, 0, withAppostr, 0);
            resetCollections(&pExistingColl, &pNewColl);
         }
//         removeRepeatingParts(*pExistingColl);
      }
   }
   while(pNode2);
   delete pNewColl;
   return pExistingColl;
}

ResultCollection* StemManager::createForms(const std_string &inString, int stemNr, int withAppostr)
{
   ResultCollection *pOutputColl=new ResultCollection;
   Result2Collection homonymColl2;
   ResultCollection tmpColl;
   ResultCollection tmpCollOrt;
   ResultCollection *pStepOutputColl, *pCurColl;
   if(stemNr>=0 && stemNr<kNumberOfStemTypes)
   {
      NodeQueue front;
      StemNode *pNode=firstStem[stemNr];
      front.push_back(pNode);
      DerivationResult *pRes=new DerivationResult(inString, pNode->value());
      pCurColl=new ResultCollection;
      pCurColl->push_back(pRes);
      homonymColl2.push_back(pCurColl);
      while(!front.empty())
      {
         pNode=front.front();
         front.pop_front();
         for(pNode=pNode->fpFirstChild; pNode; pNode=pNode->fpSibling)
         {
            front.push_back(pNode);
            std_string grammarName=grammarFromCodes(pNode->fpParent->pureValue(), pNode->pureValue());
            int nrHomonyms=homonymColl2.size();
            for(int k=0; k<nrHomonyms; k++)
            {
               pCurColl=homonymColl2[k];
               ResultCollection::iterator iter;
               for(iter=pCurColl->begin(); iter!=pCurColl->end(); iter++)
                  if((*iter)->stemCode()==pNode->fpParent->value())
                  {
                     transform(&tmpColl, *iter, grammarName, pNode->value(),
                            stemNr, pNode->hasExceptionalTransition(), withAppostr, 0);
                     break;
                  }

               int ortNr=kNoOrt;
               if((ortNr=pNode->ortography())!=kNoOrt)
               {
                  ResultCollection::iterator iter2;
                  for(iter2=tmpColl.begin(); iter2!=tmpColl.end(); iter2++)
                  {
                     int index1=tmpCollOrt.size();
                     transform(&tmpCollOrt, *iter2, fOrtography[ortNr], pNode->value(),
                               stemNr, 0, withAppostr, 0);
                     //set correct exception codes what ortography otherwise masks

                     int index2=tmpCollOrt.size();
                     for(int x=index1; x<index2; x++)
                        tmpCollOrt[x]->setExceptionCode((*iter2)->exceptionCode());
                  }
                  FlushCollection(tmpColl);
                  pStepOutputColl=&tmpCollOrt;
               }
               else
                  pStepOutputColl=&tmpColl;

               int nrStepResults=pStepOutputColl->size();
               if(nrStepResults>0)
               {
                  for(int z=1; z<nrStepResults; z++)
                  {
                     ResultCollection *pNewColl=new ResultCollection;
                     ResultCollection::iterator iter3;
                     for(iter3=pCurColl->begin(); iter3!=pCurColl->end(); iter3++)
                     {
                        DerivationResult *pExstResult=new DerivationResult(**iter3);
                        pNewColl->push_back(pExstResult);
                     }
                     pNewColl->push_back((*pStepOutputColl)[z]);
                     homonymColl2.push_back(pNewColl);
                  }
                  pCurColl->push_back((*pStepOutputColl)[0]);
               }
               tmpColl.erase(tmpColl.begin(), tmpColl.end());
               tmpCollOrt.erase(tmpCollOrt.begin(), tmpCollOrt.end());
            }
         }
      }
   }
   for(int x=0; x<homonymColl2.size(); x++)
   {
      pCurColl=homonymColl2[x];
      ResultCollection::iterator iter4;
      for(iter4=pCurColl->begin(); iter4!=pCurColl->end(); iter4++)
      {
         pOutputColl->push_back(*iter4);
      }
      pCurColl->erase(pCurColl->begin(), pCurColl->end());
   }
   Result2Collection::iterator iDel;
   for(iDel=homonymColl2.begin(); iDel!=homonymColl2.end(); iDel++)
      delete *iDel;
   homonymColl2.erase(homonymColl2.begin(), homonymColl2.end());
   return pOutputColl;
}


StemManager& StemManager::setOrtography(int ortNr, const std_string &grammarName)
{
   fOrtography[ortNr]=grammarName;
   return *this;
}

